<?php

namespace App\Http\Controllers;

use App\Mail\forgetPasswordMail;
use App\Models\Cliente;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;


class AuthController extends Controller
{
    public function auth(Request $request)
    {
        if(Auth::attempt(['email_acesso' => $request->input('email_acesso'), 'password'=>$request->input('password')]))
        {
            $user = Auth::user();

            if($user->painel === 'Administrador'){
                if($user->estado == 1){

                    
                    return redirect()->route('painel.admin')->with([
                                'toast_message'=>'Bem-vindo',
                                'toast_type' => 'success'
                                ]
                                );
                }else{
                    abort(403, 'Conta desativada, entre em contacto com a equipe de suporte!');
                }
            }
            
            if($user->painel === 'Cliente'){
                if($user->estado== 1){

                    
                    return redirect()->route('painel.cliente')->with([
                                'toast_message'=>'Bem-vindo',
                                'toast_type' => 'success'
                                ]
                                );
                }else{
                    abort(403, 'Conta desativada, entre em contacto com a equipe de suporte!');
                }
            }
        }else{
            return redirect()->back()->with([
                'toast_message'=>'Credências erradas',
                'toast_type' => 'error'
                ]
            )->onlyInput('email');
        }
        
    }

    public function change_password(Request $request){
        $usuario = Usuario::where('email_acesso', $request->email);

        $request->validate([
            'palavra_chave_actual'=> 'required|min:6',
            'nova_palavra_chave'=> 'required|min:6',
            'confirma_palavra_chave' => 'required|same:nova_palavra_chave',
        ],[
            'palavra_chave_actual.required'   => 'A palavra-chave atual é obrigatória.',
            'palavra_chave_actual.min'        => 'A palavra-chave atual deve ter no mínimo 6 caracteres.',
            
            'nova_palavra_chave.required'     => 'A nova palavra-chave é obrigatória.',
            'nova_palavra_chave.min'          => 'A nova palavra-chave deve ter no mínimo 6 caracteres.',
            
            'confirma_palavra_chave.required' => 'A confirmação da palavra-chave é obrigatória.',
            'confirma_palavra_chave.same'     => 'A confirmação deve ser igual à nova palavra-chave.',
        ]);

        if($usuario->usuario_id == Auth::user()->usuario_id ){
            if(Hash::check($request->palavra_chave_actual, $usuario->password)){
                $usuario->update([
                    'password' => $request->nova_palavra_chave
                ]);

                return redirect()->route('logout')->with([
                    'toast_message' => 'Palvra-chave alterada com sucesso! Faça o login novamente',
                    'toast_type' => 'success'
                ]);
            }else{
                return redirect()->back()->with([
                    'toast_message' => 'Palvra-chave actual não correspondente',
                    'toast_type' => 'error'
                ]);
            }
        }

        abort(403, 'Não tem autorização para acessar está página!.');
    }

    public function criar_conta_cliente(Request $request){
        $validator = Validator::make($request->all(), [
            'nome_completo' => 'required|string',
            'email_acesso' => 'required|email|unique:usuarios,email_acesso',
            'bilhete'=> 'required|unique:clientes,bilhete_cliente|regex:/^\d{9}[A-Z]{2}\d{3}$/|size:14',
            'password'=> 'required|min:6',
        ], [
            'nome_completo.required' => 'O campo nome é obrigatório.',
            'nome_completo.string' => 'O nome deve ser um texto válido.',

            'email_acesso.required' => 'O campo e-mail é obrigatório.',
            'email_acesso.email' => 'Informe um e-mail válido.',
            'email_acesso.unique' => 'Este e-mail já está em uso.',

            'bilhete.required' => 'O número do bilhete é obrigatório.',
            'bilhete.unique' => 'Este bilhete já foi cadastrado.',
            'bilhete.regex' => 'O bilhete deve seguir o formato: 9 dígitos, 2 letras maiúsculas e 3 dígitos (ex: 123456789AB123).',
            'bilhete.size' => 'O bilhete deve ter exatamente 14 caracteres.',

            'password.required' => 'A palavra-chave é obrigatória.',
            'password.min' => 'A palavra-chave deve ter no mínimo 6 caracteres.',


            'data_nascimento.required' => 'A data de nascimento é obrigatória.',
            'data_nascimento.date' => 'Informe uma data de nascimento válida.',
        ]);


        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);
        
        }

        DB::transaction(function() use($request){


            $usuario = Usuario::create([
                'email_acesso' => $request->email_acesso,
                'password' => $request->password,
                'estado' => true,
                'painel' => 'Cliente'
            ]);

             $usuario->cliente()->create([
                'nome_cliente' => $request->nome_completo,
                'email_cliente'=> $request->email_acesso,
                'bilhete_cliente'=> $request->bilhete
            ]);

        });

        return redirect()->route('login')->with([
            'toast_message' => 'Conta criada com sucesso!',
            'toast_type' => 'success'
        ]);
    }

    public function forget_password(Request $request){
        $email = $request->input('email');

        $usuario = Usuario::where('email_acesso', $email)->first();

        if(!$usuario){
            return redirect()->back()->with([
                'toast_message' => 'E-mail não encontrado',
                'toast_type' => 'error'
            ]);
        }
        
        $email_criptografado = Crypt::encryptString($usuario->email_acesso);

        //enviar email de confirmação de conta
        try{
            Mail::to($usuario->email_acesso)->send(new forgetPasswordMail($email_criptografado));
        }catch(\Exception $e){
            Log::error('Falha ao enviar e-mail: ' . $e->getMessage()); 
        }


        return redirect()->back()->with([
            'toast_message' => 'E-mail enviado',
            'toast_type' => 'success'
        ]);
    }

    public function recover_password(Request $request){
        
        $email = Crypt::decryptString($request->email);

        $usuario = Usuario::where('email_acesso', $email);


        $validator = Validator::make([
            'password'=> 'required|min:6',
            'password-confirm' => 'required|same:password',
        ],
        [
            
            'password.required'     => 'A nova palavra-chave é obrigatória.',
            'password.min'          => 'A nova palavra-chave deve ter no mínimo 6 caracteres.',
            
            'password.confirm.required' => 'A confirmação da palavra-chave é obrigatória.',
            'password.confirm.same'     => 'A confirmação deve ser igual à nova palavra-chave.',
        ]
        );

        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);
        
        }

        if($usuario){
                $usuario->update([
                    'password' => Hash::make($request->password)
                ]);

                return redirect()->route('login')->with([
                    'toast_message' => 'Palvra-chave alterada com sucesso! Faça o login novamente',
                    'toast_type' => 'success'
                ]);
        }
    }


    public function logout(Request $request){
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('login');
    }


}
