<?php

namespace App\Http\Controllers;

use App\Models\Candidatura;
use App\Models\Vaga;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class VagasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        $search = trim($request->input('search', ''));

        $vagas = Vaga::query()->when($search, function ($query, $search) {
            $query->Where('nome_vaga', 'LIKE', "%{$search}%");
        })->paginate(10)
        ->appends(['search' => $search]);

        return view('dashboard.admin.pages.vagas.vagas', ['vagas'=> $vagas]);
    }

    /**
     * Show the form for creating a new resource.
     */

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nome_vaga' => 'required',
            'numero_vaga' => 'required',
            'data_inicio' => 'required|date',
            'data_fim'=> 'required|date',
            'imagem_vaga' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'descricao_vaga'=> 'required'
            ],
            [
                'nome_vaga.required' => 'O nome do produto é obrigatório.',
                'numero_vaga.required' => 'O número de vagas é obrigatória.',

                'data_inicio.required'=>'A data de inicio é obrigatório',
                'data_inicio.date' => 'A data de inicio tem que ser uma data',

                'data_fim.required'=>'A data de fim é obrigatório',
                'data_fim.date' => 'A data de fim tem que ser uma data',

                'imagem_vaga.image' => 'O arquivo deve ser uma imagem.',
                'imagem_vaga.mimes' => 'A imagem deve estar no formato: jpeg, png, jpg, gif ou webp.',
                'imagem_vaga.max'   => 'A imagem não pode ter mais que 2MB.',


            'descricao_vaga.required' => 'A descrição da vaga é obrigatória.',
            ]
        );

        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);

        }

        $dados = $request->only([
            'nome_vaga', 'numero_vaga',
            'data_inicio', 'data_fim',
            'descricao_vaga'
        ]);


        $dados['imagem_vaga'] = null;

        if ($request->hasFile('imagem_vaga')) {


            $path = $request->file('imagem_vaga')->store('vagas', 'public');

            $dados['imagem_vaga'] = $path;

        }

        $vaga = Vaga::create([
            'nome_vaga' => $dados['nome_vaga'],
            'descricao_vaga' => $dados['descricao_vaga'],
            'numero_vaga'=> $dados['numero_vaga'],
            'imagem_vaga'=> $dados['imagem_vaga'],
            'data_inicio' => $dados['data_inicio'],
            'data_fim'=> $dados['data_fim']

        ]);

        return redirect()->back()->with([
            'toast_message' => 'Vaga adicionada com sucesso!',
            'toast_type' => 'success'
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $vaga = Vaga::find($id);

        $dados = $request->only([
            'nome_vaga', 'numero_vaga',
            'data_inicio', 'data_fim',
            'descricao_vaga'
        ]);



        if ($request->hasFile('imagem_vaga')) {

            $path = $request->file('imagem_vaga')->store('vagas', 'public');


            $dados['imagem_vaga'] = $path;

        }

        $vaga->update($dados);

        return redirect()->back()->with([
            'toast_message' => 'Vaga actualizada',
            'toast_type' => 'success'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $vaga = Vaga::find($id);

        if($vaga){
            $vaga->delete();


            return redirect()->back()->with([
                'toast_message' => 'Vaga eliminada',
                'toast_type' => 'success'
            ]);
        }
    }
}
